/*****************************************************************************
*                           Freescale Semiconductor                          *
*                                                                            *
*  Project    : AN3815 - MC56F8006 Modular Pixel Matrix                      *
*  Version    : 0.1                                                          *
*  Date       : 16/Apr/2009                                                  *
*  Authors    : Alexandre Dias, Bruno Bastos                                 *
*               Humberto Carvalho, Leonardo Mangiapelo                       *
*               Renato Frias                                                 *
*                                                                            *
* -------------------------------------------------------------------------- *
* THIS SOFTWARE IS PROVIDED BY FREESCALE "AS IS" AND ANY EXPRESSED OR        *
* IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES  *
* OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.    *
* IN NO EVENT SHALL FREESCALE OR ITS CONTRIBUTORS BE LIABLE FOR ANY DIRECT,  *
* INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES         *
* (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR         *
* SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)         *
* HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,        * 
* STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING      *
* IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF             *
* THE POSSIBILITY OF SUCH DAMAGE.                                            *
* -------------------------------------------------------------------------- *
*  Description:                                                              *
* This software complements AN3815 - Implementing a Modular High Brightness  *
* RGB LED Network. The aim of the code is to proof the concept described on  *
* the application note document. It also serves as a starting point for      * 
* applications with MC56F8006 and some of its peripherals                    *
* -------------------------------------------------------------------------- *
*                                                                            *
*  File: color_manager.c                                                     *
*                                                                            *
*   Implements the color mixing algorithm, .  								 * 
*                                                                            *
******************************************************************************/

/************************* Include Files *************************************/

#include "util.h"
#include "adc.h"
#include "color_manager.h"
#include "pwm.h"

/******************************** Prototypes *********************************/

void colorm_init(void);
void colorm_task(void);

/******************************** Variables **********************************/

//! Control variables to set the LED color.
int16_t cm_color_red, cm_color_green, cm_color_blue;
//! Store color values received from protocol manager (SCI).
uint16_t cm_desired_red, cm_desired_green, cm_desired_blue;
//! Timeout decremented each time IDLE state runs.
uint16_t cm_count;
//! Store the command to be performed by color manager task.
CM_STATUS cm_status;
//! Color manager state machine variable.
CM_STATE_MACHINE colorm_state_machine;

/*********************** Function's Implementation ***************************/

//! Initializes color manager local variables.
/*! This function should be called before "colorm_task()"*/
void colorm_init(void)
{
	cm_color_red = 0;
	cm_color_green = 0;
	cm_color_blue = 0; 
	cm_desired_red = CM_INIT_RED;
	cm_desired_green = CM_INIT_GREEN;
	cm_desired_blue = CM_INIT_BLUE;
	colorm_state_machine = CM_IDLE;
	cm_status = CM_CMD_NONE;
	cm_count = CM_TIMEOUT;
}

//! Color Manager state machine implementation.
/*! 
	This task controls the color intensity that should be lit by the High Power-LED.
	It also receives new color requests by functions that call "colorm_rcv_value".
*/
void colorm_task(void)
{
	switch (colorm_state_machine)
	{
		case CM_IDLE:
		if(cm_count-- == 0)
		{
		   colorm_state_machine = CM_ADJ_PWM;
		   cm_count = CM_TIMEOUT;	
		}
		if(cm_status != CM_CMD_NONE)
		{
			colorm_state_machine = CM_READ_CMD;
		}
		break;
		
		case CM_ADJ_PWM:
		
		if (adc_get_color_value(ADC_RED) < cm_desired_red)
		{
			cm_color_red++;
		}
		else
		{		
			if(--cm_color_red<0)
			{
			 	cm_color_red=0;
			}
		}
		if (adc_get_color_value(ADC_GREEN) < cm_desired_green)
		{
			cm_color_green++;
		}
		else
		{
			if(--cm_color_green<0)
			{
			 	cm_color_green=0;
			}
		}
		if (adc_get_color_value(ADC_BLUE) < cm_desired_blue)
		{
			cm_color_blue++;
		}
		else
		{
			if(--cm_color_blue<0)
			{
			 	cm_color_blue=0;
			}
		}
		pwm_set_values(cm_color_red,cm_color_green,cm_color_blue);
		colorm_state_machine = CM_IDLE;
		
		break;
		
		case CM_READ_CMD:
		switch(cm_status)
		{
			case CM_CMD_LIT:
			colorm_state_machine = CM_ADJ_PWM;
			break;
			
			case CM_CMD_STORE:
			colorm_state_machine = CM_IDLE;
			//to be implemented
			break;
			 
		}
		cm_status = CM_CMD_NONE;
		
		break;
		
	}
}

//! Receives commands and color values.
/*! 
	\param r the red component to be set or stored.
	\param g the green component to be set or stored.
	\param b the blue component to be set or stored.
	\param cmd the command to be executed, set or store. 
	
	Receives desired color values and commands from other functions.
*/
void colorm_rcv_value(uint8_t r, uint8_t g, uint8_t b, CM_STATUS cmd)
{
	cm_desired_red = r * CM_ADC_SCALE;
	if(cm_desired_red < CM_LED_MIN)
	{
		cm_desired_red = 0;
	}
	cm_desired_green = g * CM_ADC_SCALE;
	if(cm_desired_green < CM_LED_MIN)
	{
		cm_desired_green = 0;
	}
	cm_desired_blue = b * CM_ADC_SCALE;
	
	if(cm_desired_blue < CM_LED_MIN)
	{
		cm_desired_blue = 0;
	}
	cm_status = cmd; 
	return;
}
